/*
 * Copyright (c) 2008, 2023, Oracle and/or its affiliates. All rights reserved.
 * ORACLE PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 */

package sun.nio.fs;

import java.nio.file.*;
import java.io.IOException;

import static sun.nio.fs.WindowsConstants.*;

/**
 * Internal exception thrown when a Win32 call fails.
 */

class WindowsException extends Exception {
    static final long serialVersionUID = 2765039493083748820L;

    private int lastError;
    private String msg;

    WindowsException(int lastError) {
        this.lastError = lastError;
        this.msg = null;
    }

    WindowsException(String msg) {
        this.lastError = 0;
        this.msg = msg;
    }

    int lastError() {
        return lastError;
    }

    String errorString() {
        if (msg == null) {
            msg = WindowsNativeDispatcher.FormatMessage(lastError);
            if (msg == null) {
                msg = "Unknown error: 0x" + Integer.toHexString(lastError);
            }
        }
        return msg;
    }

    @Override
    public String getMessage() {
        return errorString();
    }

    @Override
    public Throwable fillInStackTrace() {
        // This is an internal exception; the stack trace is irrelevant.
        return this;
    }

    private IOException translateToIOException(String file, String other) {
        // not created with last error
        if (lastError() == 0)
            return new IOException(errorString());

        // handle specific cases
        if (lastError() == ERROR_FILE_NOT_FOUND || lastError() == ERROR_PATH_NOT_FOUND)
            return new NoSuchFileException(file, other, null);
        if (lastError() == ERROR_FILE_EXISTS || lastError() == ERROR_ALREADY_EXISTS)
            return new FileAlreadyExistsException(file, other, null);
        if (lastError() == ERROR_ACCESS_DENIED)
            return new AccessDeniedException(file, other, null);

        // fallback to the more general exception
        return new FileSystemException(file, other, errorString());
    }

    void rethrowAsIOException(String file) throws IOException {
        IOException x = translateToIOException(file, null);
        throw x;
    }

    void rethrowAsIOException(WindowsPath file, WindowsPath other) throws IOException {
        String a = (file == null) ? null : file.getPathForExceptionMessage();
        String b = (other == null) ? null : other.getPathForExceptionMessage();
        IOException x = translateToIOException(a, b);
        throw x;
    }

    void rethrowAsIOException(WindowsPath file) throws IOException {
        rethrowAsIOException(file, null);
    }

    IOException asIOException(WindowsPath file) {
        return translateToIOException(file.getPathForExceptionMessage(), null);
    }

}
